<?php
// Handles DB table creation, folder setup, and activation hook.

defined('ABSPATH') || exit;

function spdfed_activate_plugin() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'spdfed_logs';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        email varchar(100) NOT NULL,
        submitter_name varchar(255) NULL DEFAULT NULL, /* Ensure consistent definition */
        file_name varchar(255) NOT NULL,
        downloaded_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        post_id mediumint(9) NULL DEFAULT NULL,
        occurred_local_date DATE NULL,
        email_hash VARBINARY(32) NULL,
        referrer varchar(255) NULL,
        ua_hash VARBINARY(32) NULL,
        ip_hash VARBINARY(32) NULL,
        PRIMARY KEY  (id),
        KEY email (email(100)),
        KEY file_name (file_name(191)),
        KEY analytics_lookup (occurred_local_date, post_id),
        KEY post_date (post_id, occurred_local_date)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
    
    // Set database version for future upgrades
    update_option('spdfed_db_version', '2.0');

    // Verify table creation
    if ($wpdb->last_error) {
        return false;
    }

    // Verify table exists
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
    $table_check = $wpdb->get_var(
        $wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $wpdb->esc_like($table_name)
        )
    );

    if (!$table_check) {
        return false;
    }

    // Create private uploads directory
    $upload_dir = wp_upload_dir();
    $dir = trailingslashit($upload_dir['basedir']) . 'email-gated-downloads';
    if (!file_exists($dir)) {
        if (!wp_mkdir_p($dir)) {
            return false;
        }
        
        // Initialize WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        
        // Create index.php for security
        $wp_filesystem->put_contents(
            trailingslashit($dir) . 'index.php',
            '<?php // Silence is golden',
            FS_CHMOD_FILE
        );
        
        // Create .htaccess to prevent direct access
        $wp_filesystem->put_contents(
            trailingslashit($dir) . '.htaccess',
            'deny from all',
            FS_CHMOD_FILE
        );
    }

    // Set default options
    add_option('spdfed_enable_gdpr', 1);
    add_option('spdfed_gdpr_message', 'I consent to my email being stored for the purposes of this download.');

    // Call the function to register CPTs and flush rules
    if (function_exists('spdfed_rewrite_flush_on_activation')) {
        spdfed_rewrite_flush_on_activation();
    }

    return true;
}

/**
 * Check if plugin is properly installed
 * @return bool
 */
function spdfed_check_installation() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'spdfed_logs';
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
    $table_exists = $wpdb->get_var(
        $wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $wpdb->esc_like($table_name)
        )
    );

    if (!$table_exists) {
        return spdfed_activate_plugin();
    }

    return true;
}

/**
 * Upgrade database schema if needed
 */
function spdfed_maybe_upgrade_database() {
    $current_version = get_option('spdfed_db_version', '1.0');
    
    if (version_compare($current_version, '2.0', '<')) {
        spdfed_upgrade_to_version_2();
    }
}

/**
 * Upgrade to version 2.0 - adds analytics columns
 */
function spdfed_upgrade_to_version_2() {
    global $wpdb;
    
    // Security: Table name is properly prefixed and escaped
    $table_name = $wpdb->prefix . 'spdfed_logs';
    
    // Security: Whitelist of allowed column definitions (hardcoded, not user input)
    // ALTER TABLE DDL cannot use prepare() for column definitions as SQL doesn't support placeholders there
    // These are hardcoded strings we control, not dynamic or user-supplied values
    $columns_to_add = [
        'post_id' => 'ADD COLUMN post_id mediumint(9) NULL DEFAULT NULL',
        'occurred_local_date' => 'ADD COLUMN occurred_local_date DATE NULL',
        'email_hash' => 'ADD COLUMN email_hash VARBINARY(32) NULL',
        'referrer' => 'ADD COLUMN referrer varchar(255) NULL',
        'ua_hash' => 'ADD COLUMN ua_hash VARBINARY(32) NULL',
        'ip_hash' => 'ADD COLUMN ip_hash VARBINARY(32) NULL'
    ];
    
    foreach ($columns_to_add as $column => $sql_fragment) {
        // Check if column exists (schema upgrade operation)
        // Use prepare() for the column name check
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SHOW COLUMNS FROM `{$wpdb->prefix}spdfed_logs` LIKE %s",
            $column
        ));
        
        if (empty($column_exists)) {
            // Security: $sql_fragment is from our hardcoded whitelist above, not user input
            // ALTER TABLE DDL cannot use prepare() with placeholders for column definitions
            // Table name uses wpdb->prefix which is safe
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQL.NotPrepared
            $wpdb->query("ALTER TABLE `{$wpdb->prefix}spdfed_logs` {$sql_fragment}");
        }
    }
    
    // Add indexes (schema upgrade operation)
    // Security: Index names and column names are hardcoded, table name uses wpdb->prefix which is safe
    // ALTER TABLE for indexes cannot use prepare() as SQL doesn't support placeholders for DDL
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQL.NotPrepared
    $wpdb->query("ALTER TABLE `{$wpdb->prefix}spdfed_logs` ADD INDEX analytics_lookup (occurred_local_date, post_id)");
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQL.NotPrepared
    $wpdb->query("ALTER TABLE `{$wpdb->prefix}spdfed_logs` ADD INDEX post_date (post_id, occurred_local_date)");
    
    // Populate occurred_local_date for existing records (one-time upgrade operation)
    // Security: All column names are hardcoded, table name uses wpdb->prefix which is safe
    // This is a data migration query with no user input
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.PreparedSQL.NotPrepared
    $wpdb->query("UPDATE `{$wpdb->prefix}spdfed_logs` SET occurred_local_date = DATE(CONVERT_TZ(downloaded_at, 'UTC', @@session.time_zone)) WHERE occurred_local_date IS NULL AND downloaded_at IS NOT NULL");
    
    // Clear analytics cache after schema upgrade
    wp_cache_delete('spdfed_has_analytics_columns');
    
    update_option('spdfed_db_version', '2.0');
}

/**
 * Check if the analytics columns exist in the logs table
 */
function spdfed_table_has_analytics_columns() {
    global $wpdb;
    static $has_analytics = null;
    
    if ($has_analytics === null) {
        // Check cache first
        $cache_key = 'spdfed_has_analytics_columns';
        $has_analytics = wp_cache_get($cache_key);
        
        if (false === $has_analytics) {
            // Use prepare() for the LIKE comparison, table name uses wpdb->prefix which is safe
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $columns = $wpdb->get_results($wpdb->prepare(
                "SHOW COLUMNS FROM `{$wpdb->prefix}spdfed_logs` LIKE %s",
                'occurred_local_date'
            ));
            $has_analytics = !empty($columns);
            wp_cache_set($cache_key, $has_analytics, '', 3600); // Cache for 1 hour
        }
    }
    
    return $has_analytics;
}

/**
 * Migrate files from old directory structure to new WordPress-compliant structure
 * Runs once per installation
 */
function spdfed_migrate_files_to_new_directory() {
    // Check if migration has already been completed
    if (get_option('spdfed_files_migrated', false)) {
        return;
    }
    
    // Old directory path (hardcoded for migration purposes only)
    $old_dir = WP_CONTENT_DIR . '/private-uploads/';
    
    // New directory path
    $upload_dir = wp_upload_dir();
    $new_dir = trailingslashit($upload_dir['basedir']) . 'email-gated-downloads/';
    
    // Check if old directory exists
    if (!is_dir($old_dir)) {
        // No old directory, nothing to migrate
        update_option('spdfed_files_migrated', true);
        return;
    }
    
    // Create new directory if it doesn't exist
    if (!file_exists($new_dir)) {
        if (!wp_mkdir_p($new_dir)) {
            // Log error but don't block - will try again next time
            // Only log if WP_DEBUG_LOG is enabled (respects WordPress debug settings)
            if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for migration diagnostics
                error_log('Email Gated Downloads: Failed to create new directory during migration: ' . $new_dir);
            }
            return;
        }
        
        // Initialize WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        
        // Create security files
        $wp_filesystem->put_contents(
            $new_dir . 'index.php',
            '<?php // Silence is golden',
            FS_CHMOD_FILE
        );
        $wp_filesystem->put_contents(
            $new_dir . '.htaccess',
            'deny from all',
            FS_CHMOD_FILE
        );
    }
    
    // Get all files from old directory (excluding security files)
    $files = array_diff(scandir($old_dir), ['.', '..', '.htaccess', 'index.php']);
    
    $migrated_count = 0;
    $failed_count = 0;
    
    foreach ($files as $file) {
        $old_file_path = $old_dir . $file;
        $new_file_path = $new_dir . $file;
        
        // Only migrate files, not directories
        if (is_file($old_file_path)) {
            // Copy file to new location
            if (copy($old_file_path, $new_file_path)) {
                $migrated_count++;
                // Optionally delete the old file after successful copy
                // Commented out for safety - you can manually delete old directory after verifying migration
                // @unlink($old_file_path);
            } else {
                $failed_count++;
                // Only log if WP_DEBUG_LOG is enabled (respects WordPress debug settings)
                if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for migration diagnostics
                    error_log('Email Gated Downloads: Failed to migrate file: ' . $file);
                }
            }
        }
    }
    
    // Mark migration as complete
    update_option('spdfed_files_migrated', true);
    
    // Log migration results (only if WP_DEBUG_LOG is enabled)
    if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
        if ($migrated_count > 0) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for migration diagnostics
            error_log(sprintf('Email Gated Downloads: Successfully migrated %d file(s) from old directory to new directory.', $migrated_count));
        }
        
        if ($failed_count > 0) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for migration diagnostics
            error_log(sprintf('Email Gated Downloads: Failed to migrate %d file(s).', $failed_count));
        }
    }
}

// Add an admin notice if installation check fails
add_action('admin_init', function() {
    if (!spdfed_check_installation()) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error">
                <p>Secure PDF Email Download plugin installation appears to be incomplete. Please deactivate and reactivate the plugin.</p>
            </div>
            <?php
        });
    } else {
        // Check for database upgrades
        spdfed_maybe_upgrade_database();
        
        // Check for file migration from old directory structure
        spdfed_migrate_files_to_new_directory();
    }
});
